<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreBookingRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return array_merge($this->guestRules(), [
            'travelers' => 'required|array|min:1',

            'travelers.*.*.type' => 'required|string|in:adult,child,infant',
            'travelers.*.*.name' => 'required|string|max:255',
            'travelers.*.*.phone' => ['required_if:travelers.*.*.type,adult', 'nullable', 'string', 'max:255'],
            'travelers.*.*.date_of_birth' => [
                'required',
                'date',
                'before:today',
                function ($attribute, $value, $fail) {
                    $type = explode('.', $attribute)[2];
                    $age = now()->diffInYears($value);

                    if ($type === 'adult' && $age < 18) {
                        $fail('Adult must be at least 18 years old.');
                    } elseif ($type === 'child' && ($age < 2 || $age >= 18)) {
                        $fail('Child age must be between 2 and 17 years.');
                    } elseif ($type === 'infant' && $age >= 2) {
                        $fail('Infant must be under 2 years old.');
                    }
                },
            ],

            'payment_method' => 'required|string|max:255|in:online,cash',
        ]);
    }

    public function guestRules(): array
    {
        if (auth()->check()) {
            return [
                'user_id' => 'required|string|max:255|exists:users,id',
            ];
        }

        return [
            'name' => 'required|string|max:255',
            'email' => 'required|email:dns|max:255',
            'phone' => 'required|string|max:255',
            'address' => 'nullable|string|max:255',
            'address_one' => 'nullable|string|max:255',
            'address_two' => 'nullable|string|max:255',
            'city_id' => 'required|string|max:255|exists:cities,id',
            'state_id' => 'required|string|max:255|exists:states,id',
            'country_id' => 'required|string|max:255|exists:countries,id',
            'postal' => 'required|string|max:255',
            'message' => 'nullable|string|max:255',
        ];
    }
}
